/**************************** XS ********************************************
Copyright (C) 2000-2023  P. Bergman

This program is free software; you can redistribute it and/or modify it under 
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include <Xm/Xm.h>

#include "class.h"
#include "defines.h"
#include "global_structs.h"

/* #define DEBUG */

extern VIEW    *vP;

int nst=0, current=0;
CLASS *st = NULL;

void send_line(char *);

static string buf;

static void c4_to_int(char *c4, int *i)
{
    union {
        char c[4];
        int i;
    } b;
    
    if (!c4) return;
    if (!i) return;
    
    memcpy(b.c, c4, 4);
    *i = b.i;
}

/*
static void c8_to_int(char *c8, int *i)
{
    union {
        char c[8];
        int i;
    } b;
    
    if (!c8) return;
    if (!i) return;
    
    memcpy(b.c, c8, 8);
    *i = b.i;
}
*/


static void c8_to_ll(char *c8, long long *i)
{
    union {
        char c[8];
        long long i;
    } b;
    
    if (!c8) return;
    if (!i) return;
    
    memcpy(b.c, c8, 8);
    *i = b.i;
}


static void c4_to_float(char *c4, float *f)
{
    union {
        char c[4];
        float f;
    } b;
    
    if (!c4) return;
    if (!f) return;
    
    memcpy(b.c, c4, 4);
    *f = b.f;
}

static void c8_to_double(char *c8, double *d)
{
    union {
        char c[8];
        double d;
    } b;
    
    if (!c8) return;
    if (!d) return;
    
    memcpy(b.c, c8, 8);
    *d = b.d;
}


static void cn_to_char(char *cn, char *c, int n)
{
    
    if (!cn) return;
    if (!c) return;
    
    memcpy(c, cn, n);
    c[n] = '\0';
}

static void fill_class_table(CLASS *o, char *i)
{
    int n;
   
    if (!o) return;
    if (!i) return;
    
    n = 0;
    c4_to_int(&i[n], &(o->xbloc));    n += 4;
    c4_to_int(&i[n], &(o->xnum));     n += 4;
    c4_to_int(&i[n], &(o->xver));     n += 4;
    cn_to_char(&i[n], o->xsourc, 12); n += 12;
    cn_to_char(&i[n], o->xline, 12);  n += 12;
    cn_to_char(&i[n], o->xtel, 12);   n += 12;
    c4_to_int(&i[n], &(o->xdobs));    n += 4;
    c4_to_int(&i[n], &(o->xdred));    n += 4;
    c4_to_float(&i[n], &(o->xoff1));  n += 4;
    c4_to_float(&i[n], &(o->xoff2));  n += 4;
    cn_to_char(&i[n], o->xtype, 4);   n += 4;
    c4_to_int(&i[n], &(o->xkind));    n += 4;
    c4_to_int(&i[n], &(o->xqual));    n += 4;
    c4_to_int(&i[n], &(o->xscan));    n += 4;
    c4_to_int(&i[n], &(o->xposa));    n += 4;
    cn_to_char(&i[n], o->xfront, 8);  n += 8;
    cn_to_char(&i[n], o->xback, 8);   n += 8;
    cn_to_char(&i[n], o->xproc, 8);   n += 8;
    cn_to_char(&i[n], o->xproj, 8);   n += 8;
    cn_to_char(&i[n], o->unused, 24); n += 24;
}

static void fill_class_v2_table(CLASS *o, char *i)
{
    int n;
   
    if (!o) return;
    if (!i) return;
    
    n = 0;
    c8_to_ll(&i[n], &(o->xlbloc));    n += 8;
    c4_to_int(&i[n], &(o->xword));    n += 4;
    c8_to_ll(&i[n], &(o->xlnum));     n += 8;
    c4_to_int(&i[n], &(o->xver));     n += 4;
    cn_to_char(&i[n], o->xsourc, 12); n += 12;
    cn_to_char(&i[n], o->xline, 12);  n += 12;
    cn_to_char(&i[n], o->xtel, 12);   n += 12;
    c4_to_int(&i[n], &(o->xdobs));    n += 4;
    c4_to_int(&i[n], &(o->xdred));    n += 4;
    c4_to_float(&i[n], &(o->xoff1));  n += 4;
    c4_to_float(&i[n], &(o->xoff2));  n += 4;
    cn_to_char(&i[n], o->xtype, 4);   n += 4;
    c4_to_int(&i[n], &(o->xkind));    n += 4;
    c4_to_int(&i[n], &(o->xqual));    n += 4;
    c4_to_int(&i[n], &(o->xposa));    n += 4;
    c8_to_ll(&i[n], &(o->xlscan));    n += 8;
    c4_to_int(&i[n], &(o->xsubs));    n += 4;
    
    o->xbloc = (int)o->xlbloc;
    o->xnum = (int)o->xlnum;
    o->xscan = (int)o->xlscan;
/*    
#ifdef DEBUG
    printf("Record where obs is to be found: block=%lld.\n", o->xlbloc);
    printf("The word where obs start in this record: Word=%d\n", o->xword);
    printf("Observation number: Num=%lld\n", o->xlnum);
    printf("Scan %lld: %s %s %s\n", o->xlscan, o->xsourc, o->xline, o->xtel);
#endif
*/
}

static void fill_class_v2_entrydesc(CLASS_SECTION_v2 *o, char *edesc)
{
    int a = 0, m, nmax;
    
    if (!o) return;
    if (!edesc) return;
    
    memcpy(o->ident, &edesc[a], 4);	   a += 4;
    c4_to_int(&edesc[a], &(o->version));   a += 4;
    c4_to_int(&edesc[a], &(o->nsec));      a += 4;
    c8_to_ll(&edesc[a], &(o->nword));      a += 8;
    c8_to_ll(&edesc[a], &(o->adata));      a += 8;
    c8_to_ll(&edesc[a], &(o->ldata));      a += 8;
    c8_to_ll(&edesc[a], &(o->xnum));	   a += 8;
    
    nmax = o->nsec;
    if (nmax >= MAX_CLASS_SECT) nmax=MAX_CLASS_SECT;
    
    for (m=0; m<nmax; m++) {
      c4_to_int(&edesc[a], &(o->sec_cod[m])); a += 4;
    }
    for (m=0; m<nmax; m++) {
      c8_to_ll(&edesc[a], &(o->sec_len[m])); a += 8;
    }
    for (m=0; m<nmax; m++) {
      c8_to_ll(&edesc[a], &(o->sec_adr[m])); a += 8;
    }
#ifdef DEBUG
    printf("a=%d\n", a);
    printf("  ident   '%s' Identifier of Entry Description.\n", o->ident);
    printf("  version '%d' Observations version.\n", o->version);
    printf("  nsec    '%d' Number of sections.\n", o->nsec);
    printf("  nword   '%lli' Length of this entry [words].\n", o->nword);
    printf("  adata   '%lli' Data address [word].\n", o->adata);
    printf("  ldata   '%lli' Data length [words].\n", o->ldata);
    printf("  xnum    '%lli' Entry number.\n", o->xnum);
    for (m=0; m<nmax; m++) {
      printf("  %3d len=%lld words at addr %lld\n", o->sec_cod[m], o->sec_len[m], o->sec_adr[m]);
    }
#endif
}

static void fill_class_obs(CLASS_SECTION *o, char *i, int size)
{    
    int n, nmax, a;
    
    if (!o) return;
    if (!i) return;
    
    if (size < 36) return;
        
    a = 0;
    memcpy(o->ident, &i[a], 4);     a += 4;
    c4_to_int(&i[a], &(o->nbl));    a += 4;
    c4_to_int(&i[a], &(o->bytes));  a += 4;
    c4_to_int(&i[a], &(o->adr));    a += 4;
    c4_to_int(&i[a], &(o->nhead));  a += 4;
    c4_to_int(&i[a], &(o->len));    a += 4;
    c4_to_int(&i[a], &(o->ientry)); a += 4;
    c4_to_int(&i[a], &(o->nsec));   a += 4;
    c4_to_int(&i[a], &(o->obsnum)); a += 4;
    
    nmax = o->nsec;
    if (nmax >= 4) nmax=4;
    
    for (n=0; n<nmax; n++) {
        if (a+4 >= size) return;
        c4_to_int(&i[a], &(o->sec_cod[n])); a += 4;
    }
    for (n=0; n<nmax; n++) {
        if (a+4 >= size) return;
        c4_to_int(&i[a], &(o->sec_len[n])); a += 4;
    }
    for (n=0; n<nmax; n++) {
        if (a+4 >= size) return;
        c4_to_int(&i[a], &(o->sec_adr[n])); a += 4;
    }
}

/*
static int check_block(CLASS *c, int nst, int block_no)
{
    int n;
    
    for (n=1; n<nst; n++) {
        if (c[n].xbloc == block_no) return n;
    }
    
    return 0;
}
*/

static void fill_class_header(int code, int first, int len, char *s, int size, CLASS *c)
{
    int n;
    
    if (!s) return;
    if (!c) return;
    
    n = (first-1)*4;
    if (code == -2) { /* General */
	c8_to_double(&s[n], &(c->g.ut));  n += 8;
	c8_to_double(&s[n], &(c->g.st));  n += 8;
	c4_to_float(&s[n], &(c->g.az));   n += 4;
	c4_to_float(&s[n], &(c->g.el));   n += 4;
	c4_to_float(&s[n], &(c->g.tau));  n += 4;
	c4_to_float(&s[n], &(c->g.tsys)); n += 4;
	c4_to_float(&s[n], &(c->g.time)); n += 4;
	
	if (len > 9) {
	  c4_to_int(&s[n], &(c->g.xunit)); n += 4;
	} else
	  c->g.xunit = 0;
	
#ifdef DEBUG
	printf("    -2  UT=%f %f   Az,El=%f,%f  time=%f\n", c->g.ut, c->g.st, c->g.az, c->g.el, c->g.time);
#endif
    } else if (code == -3) { /* Position */
      if (len == 17) {
	cn_to_char(&s[n], c->p.source, 12); n += 12;
	c4_to_float(&s[n], &(c->p.epoch));  n += 4;
 	c8_to_double(&s[n], &(c->p.lam));   n += 8;
	c8_to_double(&s[n], &(c->p.bet));   n += 8;
	c4_to_float(&s[n], &(c->p.lamof));  n += 4;
	c4_to_float(&s[n], &(c->p.betof));  n += 4;
	c4_to_int(&s[n], &(c->p.proj));     n += 4;
 	c8_to_double(&s[n], &(c->p.sl0p));  n += 8;
 	c8_to_double(&s[n], &(c->p.sb0p));  n += 8;
 	c8_to_double(&s[n], &(c->p.sk0p));  n += 8;
      } else { /* New CLASS positional section from 2015 */
	cn_to_char(&s[n], c->p.source, 12); n += 12;
 	c4_to_int(&s[n], &(c->p.system));   n += 4;
	c4_to_float(&s[n], &(c->p.epoch));  n += 4;
	c4_to_int(&s[n], &(c->p.proj));     n += 4;
 	c8_to_double(&s[n], &(c->p.lam));   n += 8;
	c8_to_double(&s[n], &(c->p.bet));   n += 8;
	c8_to_double(&s[n], &(c->p.projang));   n += 8;
	c4_to_float(&s[n], &(c->p.lamof));  n += 4;
	c4_to_float(&s[n], &(c->p.betof));  n += 4;
      }
	
#ifdef DEBUG
      printf("    -3  '%s' %f Coord:%f,%f (%f,%f) %4d\n", c->p.source,
	       c->p.epoch, c->p.lam, c->p.bet, c->p.lamof, c->p.betof, c->p.proj);
#endif
    } else if (code == -4) { /* Spectroscopic section */
      /* Only version 1.3 supported here len=17 */
	cn_to_char(&s[n], c->s.line, 12);     n += 12;
  	c8_to_double(&s[n], &(c->s.restf));   n += 8;
	c4_to_int(&s[n], &(c->s.nchan));      n += 4;
	c4_to_float(&s[n], &(c->s.rchan));    n += 4;
	c4_to_float(&s[n], &(c->s.fres));     n += 4;
	c4_to_float(&s[n], &(c->s.foff));     n += 4;
	c4_to_float(&s[n], &(c->s.vres));     n += 4;
	c4_to_float(&s[n], &(c->s.voff));     n += 4;
	c4_to_float(&s[n], &(c->s.bad));      n += 4;
  	c8_to_double(&s[n], &(c->s.image));   n += 8;
	c4_to_int(&s[n], &(c->s.vtype));      n += 4;
  	c8_to_double(&s[n], &(c->s.doppler)); n += 8;
	
#ifdef DEBUG
	printf("    -4  '%s' %f i%f (%f %f) n=%d r=%f v=%f %f %f\n", c->s.line,
	       c->s.restf, c->s.image, c->s.fres, c->s.foff, c->s.nchan, c->s.rchan, c->s.vres, c->s.voff, c->s.doppler);
#endif
    } else if (code == -5) { /* Baseline info section (for spectra or drifts) */
        /* Not supported, silently ignored */
    } else if (code == -6) { /* Scan numbers of initial observations */
        /* Not supported , silently ignored*/
    } else if (code == -7) { /* Default plotting limits */
        /* Not supported, silently ignored */
    } else if (code == -8) { /* Switching information */
        /* Not supported, silently ignored */
    } else if (code == -9) { /* Gauss fit results */
        /* Not supported, silently ignored */
    } else if (code == -10) { /* Continuum drifts section */
  	c8_to_double(&s[n], &(c->u.freq));  n += 8;
	c4_to_float(&s[n], &(c->u.width));  n += 4;
	c4_to_int(&s[n], &(c->u.npoin));    n += 4;
	c4_to_float(&s[n], &(c->u.rpoin));  n += 4;
	c4_to_float(&s[n], &(c->u.tref));   n += 4;
	c4_to_float(&s[n], &(c->u.aref));   n += 4;
	c4_to_float(&s[n], &(c->u.apos));   n += 4;
	c4_to_float(&s[n], &(c->u.tres));   n += 4;
	c4_to_float(&s[n], &(c->u.ares));   n += 4;
	c4_to_float(&s[n], &(c->u.bad));    n += 4;
	c4_to_int(&s[n], &(c->u.ctype));    n += 4;
  	c8_to_double(&s[n], &(c->u.cimag)); n += 8;
	c4_to_float(&s[n], &(c->u.colla));  n += 4;
	c4_to_float(&s[n], &(c->u.colle));  n += 4;
	
#ifdef DEBUG
	printf("   -10  f=%f %f  n=%d r=%f\n", c->u.freq, c->u.width, c->u.npoin, c->u.rpoin);
	printf("   -10  t=%f %f  a=%f %f\n", c->u.tref, c->u.tres, c->u.aref, c->u.ares);
#endif
    } else if (code == -14) { /* Calibration */
	c4_to_float(&s[n], &(c->c.beeff));    n += 4;
	c4_to_float(&s[n], &(c->c.foeff));    n += 4;
	c4_to_float(&s[n], &(c->c.gaini));    n += 4;
	c4_to_float(&s[n], &(c->c.h2omm));    n += 4;
	c4_to_float(&s[n], &(c->c.pamb));     n += 4;
	c4_to_float(&s[n], &(c->c.tamb));     n += 4;
	c4_to_float(&s[n], &(c->c.tatms));    n += 4;
	c4_to_float(&s[n], &(c->c.tchop));    n += 4;
	c4_to_float(&s[n], &(c->c.tcold));    n += 4;
	c4_to_float(&s[n], &(c->c.taus));     n += 4;
	c4_to_float(&s[n], &(c->c.taui));     n += 4;
	c4_to_float(&s[n], &(c->c.tatmi));    n += 4;
	c4_to_float(&s[n], &(c->c.trec));     n += 4;
	c4_to_int(&s[n], &(c->c.cmode));      n += 4;
	c4_to_float(&s[n], &(c->c.atfac));    n += 4;
	c4_to_float(&s[n], &(c->c.alti));     n += 4;
	c4_to_float(&s[n], &(c->c.count[0])); n += 4;
	c4_to_float(&s[n], &(c->c.count[1])); n += 4;
	c4_to_float(&s[n], &(c->c.count[2])); n += 4;
	c4_to_float(&s[n], &(c->c.lcalof));   n += 4;
	c4_to_float(&s[n], &(c->c.bcalof));   n += 4;
  	c8_to_double(&s[n], &(c->c.geolong)); n += 8;
  	c8_to_double(&s[n], &(c->c.geolat));  n += 8;
        
#ifdef DEBUG
	printf("   -14 %d %f %f %f trec=%f  (%f,%f) Site:(%f,%f, %f)\n", n/4, c->c.h2omm, c->c.tamb, c->c.pamb, c->c.trec,
	       c->c.lcalof, c->c.bcalof, c->c.geolong, c->c.geolat, c->c.alti);
#endif
    } else {
        sprintf(buf, "Cannot handle CLASS section code %d yet. Sorry.", code);
	send_line(buf);
    }
}

static void fill_class_data(CLASS_SECTION *cs, char *s, int size, CLASS *c)
{
    int i, n, ndata;
    char *p;
    
    n = 4*(cs->nhead - 1);
    
    if (c->xkind == 1)
      ndata = c->u.npoin;
    else
      ndata = c->s.nchan;
    
    c->ndata = ndata;
    
    c->data = (float *)malloc(ndata * sizeof(float));
    
    if (!c->data) {
      c->ndata = 0;
      return;
    }
      
    p = &s[n];
    
    for (i=0; i<ndata; i++) {
	c4_to_float(p, &(c->data[i]));
        p += 4;
    }
    
#ifdef DEBUG
    if (ndata > 30)
        printf("    -D1 %d %d  %f %f %f %f ... %f %f\n", ndata, n,
           c->data[0], c->data[10], c->data[20], c->data[30],
	   c->data[ndata-2], c->data[ndata-1]);
#endif
}

static void fill_class_v2_data(CLASS_SECTION_v2 *cs, char *s, int size, CLASS *c)
{
    int i, ndata;
    char *p;
    
    if (c->xkind == 1)
      ndata = c->u.npoin;
    else
      ndata = c->s.nchan;
    
    c->ndata = ndata;
    
    c->data = (float *)malloc(ndata * sizeof(float));
    
    if (!c->data) {
        c->ndata = 0;
        return;
    }
      
    p = &s[0];
    
    for (i=0; i<ndata; i++) {
	c4_to_float(p, &(c->data[i]));
        p += 4;
    }
    
#ifdef DEBUG
    if (ndata > 30)
        printf("    -D2 %d  %f %f %f %f ... %f %f\n", ndata,
           c->data[0], c->data[10], c->data[20], c->data[30],
	   c->data[ndata-2], c->data[ndata-1]);
#endif
}

void set_classfile_type(char *file, CLASS_INFO *i)
{
    int len, record_length=0;
    FILE *fp;
    static char type[4], reclen[4];
    
    if (!i) return;
    
    i->type = 0;
    
    fp = fopen(file, "r");                  /* open the file for reading  */
    if (!fp) {                              /* test for failure           */
        sprintf(buf, "Cannot open file '%s'.\n", file);
	send_line(buf);
        return;
    }
    
    len = fread(type, sizeof(char), 4, fp);
    if (len != 4) {
        sprintf(buf, "Cannot read %d byte type from file '%s'.\n", 4, file);
	send_line(buf);
        fclose(fp);
	return;
    }
    
    if (strncmp(type, "1A", 2) == 0) {
        i->type = 1;
        i->reclen = 128;
    } else if (strncmp(type, "2A", 2) == 0) {
        i->type = 2;
        len = fread(reclen, sizeof(char), 4, fp);
        if (len != 4) {
            sprintf(buf, "Cannot read %d byte reclen from file '%s'.\n", 4, file);
	    send_line(buf);
            fclose(fp);
	    return;
        }
        c4_to_int(reclen, &record_length);
        i->reclen = record_length;
    } else {
        fclose(fp);
	return;
    }
    
    fclose(fp);
    
 #ifdef DEBUG
    printf("Found CLASS file of type='%s' and record length=%d words.\n", type, record_length);
 #endif
    
    return;
}

FILE *get_classfile_descriptor(char *file, CLASS_INFO *i)
{
    int len, rec_size, typeok=0, mex, m, a;
    char *rec = NULL;
    FILE *fp;
    
    rec_size = 4 * i->reclen;
    rec = (char *)malloc(rec_size);
    if (!rec) {
        sprintf(buf, "Cannot allocate record of size %d.\n", rec_size);
	send_line(buf);
        return NULL;
    }
    
    fp = fopen(file, "r");                  /* open the file for reading  */
    if (!fp) {                              /* test for failure           */
        sprintf(buf, "Cannot open file '%s'.\n", file);
	send_line(buf);
	free(rec);
        return NULL;
    }
    
    len = fread(rec, sizeof(char), rec_size, fp);
    if (len != rec_size) {
        sprintf(buf, "Cannot read %d byte record from file '%s'.\n", rec_size, file);
	send_line(buf);
        fclose(fp);
	free(rec);
	return NULL;
    }
    
    if (i) {
	memcpy(i->fic, &rec[0], 4);
	i->fic[3] = '\0';
	/* Version 1 file descriptors */
	if (strncmp(i->fic, "1A", 2) == 0) {
	  typeok = 1;
	  c4_to_int(&rec[4], &(i->nextbl));
	  c4_to_int(&rec[8], &(i->nie));
	  c4_to_int(&rec[12], &(i->nex));
	  c4_to_int(&rec[16], &(i->first));
#ifdef DEBUG
	  printf("fic '%s'  (Type of data '1A' -> IEEE)\n", i->fic);
	  printf("nextbl '%d' (Next free block)\n", i->nextbl);
	  printf("nie '%d' (No of index entries)\n", i->nie);
	  printf("nex '%d' (No of index extensions)\n", i->nex);
	  printf("first '%d' (First block of 1st index extension)\n", i->first);
#endif
        }
	/* Version 2 file descriptors */
	if (strncmp(i->fic, "2A", 2) == 0) {
	  typeok = 2;
	  c4_to_int(&rec[4], &(i->reclen));
	  c4_to_int(&rec[8], &(i->kind));
	  c4_to_int(&rec[12], &(i->vind));
	  c4_to_int(&rec[16], &(i->lind));
	  c4_to_int(&rec[20], &(i->flags));
	  
	  c8_to_ll(&rec[24], &(i->xnext));
	  c8_to_ll(&rec[32], &(i->nextrec));
 	  c4_to_int(&rec[40], &(i->nextword));
	  
 	  c4_to_int(&rec[44], &(i->lex1));
	  c4_to_int(&rec[48], &(i->nex));
	  c4_to_int(&rec[52], &(i->gex));
	  
	  mex = i->nex;
	  if (mex > 10) mex = 10;
	  a = 56;
	  for (m=0; m<10; m++) {
	    c8_to_ll(&rec[a], &(i->aex[m])); a += 8;
	  }
	  
#ifdef DEBUG
	  printf("reclen '%d' (Record length [words])\n", i->reclen);
	  printf("kind '%d' (File kind)\n", i->kind);
	  printf("vind '%d' (Index version)\n", i->vind);
	  printf("lind '%d' (Index length [words])\n", i->lind);
	  printf("flags '%d' (Bit flags)\n", i->flags);
	  printf("xnext '%lld' (Next available entry number)\n", i->xnext);
	  printf("nextrec '%lld' (Next record which contains free space [recorrd])\n", i->nextrec);
	  printf("nextword '%d' (Next free word in this record [word])\n", i->nextword);
	  printf("lex1 '%d' (Length of first extension index [entries])\n", i->lex1);
	  printf("nex '%d' (No of index extensions)\n", i->nex);
	  printf("gex '%d' (Extension growth rule)\n", i->gex);
	  for (m=0; m<mex; m++) {
	       printf("  %d   aex=%lli record\n", m, i->aex[m]);
	  }
#endif
          i->first = (int)i->lex1;
	}
	i->type = typeok;
    }
    if (i->nex != 1 && typeok == 1) {
        sprintf(buf, "Cannot read %d index extensions from file '%s' of type 1.", i->nex, file);
	send_line(buf);
        fclose(fp);
	free(rec);
	return NULL;
    }
    free(rec);
    return fp;
}

static void list_xdata(int nbl, CLASS *c, int ver)
{
    double rta(float);
    
#ifdef DEBUG
    printf("%5d 1 %7d %4d %d %12s %12s %12s %d (%+9.2e,%+9.2e) '%s' %d %d %d\n",
           nbl+1,
	   c->xbloc, c->xnum, c->xver, c->xsourc, c->xline, c->xtel,
	   c->xdobs, c->xoff1, c->xoff2,
	   c->xtype, c->xkind, c->xqual, c->xscan);
#endif
    if (ver == 1)
        sprintf(buf, "%5d 1 %7d %4d %d %12s %12s %12s %d (%+7.1f\",%+7.1f\") %d %8s",
           nbl+1,
	   c->xbloc, c->xnum, c->xver, c->xsourc, c->xline, c->xtel,
	   c->xdobs, rta(c->xoff1), rta(c->xoff2), c->xscan, c->xback);
    else
        sprintf(buf, "%5d 1 %7d %4d %d %12s %12s %12s %d (%+7.1f\",%+7.1f\") %d",
               nbl+1,
           c->xbloc, c->xnum, c->xver, c->xsourc, c->xline, c->xtel,
           c->xdobs, rta(c->xoff1), rta(c->xoff2), c->xscan);

    send_line(buf);
}

static int get_class_v1_file_listing(FILE *fp, CLASS_INFO *i)
{
    int n, len, nscan=0;
    char block[CLASS_BLK_SIZE];
    char index1[129], index2[129], index3[129], index4[129];
    CLASS obs;
    
    if (!fp) return -1;
    
    nst = i->first;
    st = (CLASS *)malloc(i->first * sizeof(CLASS));
    if (!st) {
        nst = 0;
        sprintf(buf, "Out of memory: Cannot allocate CLASS file structure.");
	    send_line(buf);
        fclose(fp);
	    return -1;
    }
#ifdef DEBUG
    printf("Allocated st[] array for %d scans.\n", nst);
#endif 
    obs.data = NULL;
    
    /* move one blk forward */
    fseek(fp, CLASS_BLK_SIZE*sizeof(char), SEEK_CUR);
    
    for (n=2; n < i->first; n++) {
        len = fread(block, sizeof(char), CLASS_BLK_SIZE, fp);
        if (len != CLASS_BLK_SIZE) {
           sprintf(buf, "Cannot read %d byte block from CLASS file.", CLASS_BLK_SIZE);
           send_line(buf);
           fclose(fp);
	       return -2;
        }
        /* nbl++; */
        memcpy(index1, &block[0], 128);
        memcpy(index2, &block[128], 128);
        memcpy(index3, &block[128+128], 128);
        memcpy(index4, &block[128+128+128], 128);
      
       index1[128] = index2[128] = index3[128] = index4[128] = '\0';
      
      fill_class_table(&obs, index1);
      if (obs.xnum > 0 && obs.xnum < nst) {
          st[obs.xnum] = obs;
          list_xdata(n, &obs, 1);
	  nscan++;
      } else
          break;
      fill_class_table(&obs, index2);
      if (obs.xnum > 0 && obs.xnum < nst) {
          st[obs.xnum] = obs;
          list_xdata(n, &obs, 1);
	  nscan++;
      } else
          break;
      fill_class_table(&obs, index3);
      if (obs.xnum > 0 && obs.xnum < nst) {
          st[obs.xnum] = obs;
          list_xdata(n, &obs, 1);
	  nscan++;
      } else
          break;
      fill_class_table(&obs, index4);
      if (obs.xnum > 0 && obs.xnum < nst) {
          st[obs.xnum] = obs;
          list_xdata(n, &obs, 1);
	  nscan++;
      } else
          break;
    }

#ifdef DEBUG    
    printf("nscan=%d  nbl=%d  i->first=%d  i->nie=%d\n", nscan, nbl, i->first, i->nie);
#endif
    
    return nscan;
}

static int get_class_v2_file_listing(FILE *fp, CLASS_INFO *i, int extno)
{
    int isize, len, j, nread, pos, extgrowth=1;
    char *index = NULL, *iptr;
    CLASS obs;
    
    if (!fp) return -1;
    
    for (j=0; j<extno; j++) {
        extgrowth *= 2;
    }
    
    pos = (i->aex[extno] - 1)*1024;
    
    isize = 4 * i->lex1 * extgrowth * i->lind;
    
    nst = i->first * extgrowth;

#ifdef DEBUG    
    printf("*** extno=%d   nst=%d  pos=%d   isize=%d\n", extno, nst, pos, isize);
#endif
    
    st = (CLASS *)malloc(nst * sizeof(CLASS));
    if (!st) {
        nst = 0;
        sprintf(buf, "Out of memory: Cannot allocate CLASS file structure.");
	send_line(buf);
        fclose(fp);
	return -1;
    }
#ifdef DEBUG
    printf("Allocated st[] array for %d scans.\n", nst);
#endif    

/* lex1 is max no of entries in the first extension block */
/* nex is no of extension blocks, starting at record aex[] */
/* next available entry is xnext, so we have valid entries between [1,xnext-1] */

/* size=4*lex1*lind should hold maximum possible index */
    index = (char *)malloc(isize * sizeof(char));
    
    if (!index) {
      sprintf(buf, "Cannot allocate CLASS index of length %d.", isize);
      send_line(buf);
      fclose(fp);
      return -1;
    }
    
    fseek(fp, 4*pos, SEEK_SET);
    len = fread(index, sizeof(char), isize, fp);
    if (len != isize) {
        sprintf(buf, "Read only %d bytes of %d from CLASS file.", len, isize);
	send_line(buf);
	free(index);
        fclose(fp);
	return -2;
    }
    
    nread = 0;
    iptr = index;
    for (j=0; j < nst; j++) { /* loop over all existing indices */
        fill_class_v2_table(&obs, iptr);
	obs.data = NULL;
        st[j] = obs;
	if (obs.xnum >= 1) {
            list_xdata(j, &obs, 2);
	    nread++;
	}
/* move pointer to next entry */
        iptr += 4*i->lind; 
    }
    free(index);
/* return the no of scans */
    return nread;
}

int get_old_class_v2_test(FILE *fp, CLASS_INFO *i)
{   
    int n, m, len, nmax, off, a, isize=0, tsize, tpos, first_extension, section_size;
    int code, first, length;
    char *index = NULL, *section = NULL, *table = NULL;
    CLASS_SECTION_v2 cs2;
    CLASS obs;

    tsize = 4 * 26;
    table = (char *)malloc(tsize * sizeof(char));
    
    tpos = 4096; 
    fseek(fp, tpos*sizeof(char), SEEK_SET);
    fread(table, sizeof(char), tsize, fp);
    
    fill_class_v2_table(&obs, table);
    
    first_extension = 4 * i->aex[0] * i->reclen;

#ifdef DEBUG    
    printf("i->lind=%d words,  i->first: %d    long long=%d  char=%d\n",
           i->lind, i->first, (int)sizeof(long long), (int)sizeof(char));
#endif
    
    for (n=0; n<1; n++) {
      off = first_extension + n;
      fseek(fp, off*sizeof(char), SEEK_SET);
      len = fread(index, sizeof(char), isize, fp);
      printf("n=%d off=%d  isize= %d  len=%d\n", n, off, isize, len);
      if (len != isize) {
        sprintf(buf, "Cannot read %d word index from %d CLASS file.", len, isize);
	send_line(buf);
	free(index);
        fclose(fp);
	return -2;
      }
      
      a = 0;
      
      memcpy(cs2.ident, &index[a], 4);        a += 4;
      c4_to_int(&index[a], &(cs2.version));   a += 4;
      c4_to_int(&index[a], &(cs2.nsec));      a += 4;
      c8_to_ll(&index[a], &(cs2.nword));      a += 8;
      c8_to_ll(&index[a], &(cs2.adata));      a += 8;
      c8_to_ll(&index[a], &(cs2.ldata));      a += 8;
      c8_to_ll(&index[a], &(cs2.xnum));       a += 8;
    
      nmax = cs2.nsec;
      if (nmax >= 10) nmax=10;
    
      for (m=0; m<nmax; m++) {
        c4_to_int(&index[a], &(cs2.sec_cod[m])); a += 4;
      }
      for (m=0; m<nmax; m++) {
        c8_to_ll(&index[a], &(cs2.sec_len[m])); a += 8;
      }
      for (m=0; m<nmax; m++) {
        c8_to_ll(&index[a], &(cs2.sec_adr[m])); a += 8;
      }

      
#ifdef DEBUG
      printf("n=%d  m=%d  a=%d\n", n, off, a);
      printf("  ident   '%s' Identifier of Entry Description.\n", cs2.ident);
      printf("  version '%d' Observations version.\n", cs2.version);
      printf("  nsec    '%d' Number of sections.\n", cs2.nsec);
      printf("  nword   '%lli' Length of this entry [words].\n", cs2.nword);
      printf("  adata   '%lli' Data address [word].\n", cs2.adata);
      printf("  ldata   '%lli' Data length [words].\n", cs2.ldata);
      printf("  xnum    '%lli' Entry number.\n", cs2.xnum);
      for (m=0; m<nmax; m++) {
        printf("  %3d len=%lld words at addr %lld\n", cs2.sec_cod[m], cs2.sec_len[m], cs2.sec_adr[m]);
      }
#endif
      /* loop over sections */
      for (m=0; m<nmax; m++) {
        section_size = 4 * cs2.sec_len[m];
        section = (char *)malloc(section_size * sizeof(char));
        if (!section) {
          sprintf(buf, "Cannot allocate section (size=%d) for CLASS file.", section_size);
	  send_line(buf);
	  free(index);
          fclose(fp);
	  return -2;
        }
        off = first_extension + 4 * cs2.sec_adr[m] - 4;
        fseek(fp, off*sizeof(char), SEEK_SET);
        len = fread(section, sizeof(char), section_size, fp);
        if (len != section_size) {
          sprintf(buf, "Cannot read %d word index from %d CLASS file.", len, section_size);
	  send_line(buf);
	  free(index);
	  free(section);
          fclose(fp);
	  return -2;
        }
	code = cs2.sec_cod[m];
	first = 1;
	length = cs2.sec_len[m];
	printf("Sect[%d]=%3d at adr %d of length %d.\n", m, code, off, section_size);
        fill_class_header(code, first, length, section, section_size, &obs);
	free(section);
      }
      st[n] = obs;
      list_xdata(n, &obs, 2);
    }
    
    if (index) free(index);
    if (table) free(table);
    
    return n;
}

int get_classfile_v2_data(FILE *fp, int nscans, CLASS_INFO *i, int extno)
{
    int n, pos, datapos, secpos, len, m, nmax;
    int section_size, code, length, datasize;
    char block[CLASS_BLK_SIZE];
    char *section, *databl;
    
    CLASS_SECTION_v2 cs2;
    CLASS *o;
    
    n = 0;
       
    while (n < nscans) {
      o = &st[n];
      pos = (o->xbloc - 1)*i->reclen + o->xword - 1;
#ifdef DEBUG
      printf("n=%4d  Rec:%d Wor:%d  -> pos=%d words\n", n, o->xbloc, o->xword, pos);
#endif
      fseek(fp, 4*pos, SEEK_SET);
      len = fread(block, sizeof(char), CLASS_BLK_SIZE, fp);
      fill_class_v2_entrydesc(&cs2, block);
      
      nmax = cs2.nsec;
      if (nmax > MAX_CLASS_SECT) nmax = MAX_CLASS_SECT;
      
      for (m=0; m<nmax; m++) {
        section_size = 4*cs2.sec_len[m];
        section = (char *)malloc(section_size * sizeof(char));
        if (!section) {
          sprintf(buf, "Cannot allocate section (size=%d) for CLASS file.", section_size);
	  send_line(buf);
          fclose(fp);
	  return -2;
        }
/* For now we read each section separately - to speed up all sections could be read in one go*/
        secpos = 4*(pos + cs2.sec_adr[m] - 1);
        fseek(fp, secpos*sizeof(char), SEEK_SET);
        len = fread(section, sizeof(char), section_size, fp);
        if (len != section_size) {
          sprintf(buf, "Cannot read %d word index from %d CLASS file.", len, section_size);
	  send_line(buf);
	  free(section);
          fclose(fp);
	  return -2;
        }
	code = cs2.sec_cod[m];
	length = cs2.sec_len[m];
#ifdef DEBUG
	printf("Sect[%d]=%3d at adr %d of length %d.\n", m, code, secpos, section_size);
#endif
        fill_class_header(code, 1, length, section, section_size, o);
	free(section);
      }
      
      /* Scan the data */
      datasize = 4*cs2.ldata;
      datapos = 4*(pos + cs2.adata - 1);
      databl = (char *)malloc(datasize * sizeof(char));
      if (!databl) {
        sprintf(buf, "Cannot allocate data block (size=%d) for CLASS file.", datasize);
	send_line(buf);
        fclose(fp);
	return -3;
      }
      fseek(fp, datapos*sizeof(char), SEEK_SET);
      len = fread(databl, sizeof(char), datasize, fp);
      if (len != datasize) {
        sprintf(buf, "Read only %d of %d bytes data block CLASS file.", len, datasize);
        send_line(buf);
	free(databl);
        fclose(fp);
        return -2;
      }
      fill_class_v2_data(&cs2, databl, datasize, o);
      free(databl);
      n++;
    }
#ifdef DEBUG
    printf("Return get_classfile_v2_data() at n = %d\n", n);
#endif
    
    return 0;
}

int get_classfile_data(FILE *fp, int curr_nbl, CLASS_INFO *i)
{
    int k, len, nbl, m, first_bl, last_bl, kmax;
    char *entry;
    
    CLASS_SECTION cobs;
    CLASS *c;
        
    for (m=0; m<nst-1; m++) {
        c = &st[m+1];
	first_bl = c->xbloc;
	if (m < nst-2)
	  last_bl = st[m+2].xbloc - 1;
	else
	  last_bl = i->nextbl - 1;
	
	nbl = last_bl - first_bl + 1;
#ifdef DEBUG
        printf("Scan=%4d  %d  nbl=%3d  block range=[%5d,%5d]\n", m+1, c->xnum, nbl, first_bl, last_bl);
#endif        
	entry = (char *)malloc(nbl * 4 * 128 * sizeof(char));
	if (!entry) {
	  sprintf(buf, "CLASS: Not enough memory to allocate %d entry.", nbl);
	  send_line(buf);
	  fclose(fp);
	  return 1;
	}
	
        fseek(fp, (first_bl - 1)* 4 * 128, SEEK_SET);
        len = fread(entry, sizeof(char), nbl*4*128, fp);
	if (len != nbl*4*128) {
	  sprintf(buf, "CLASS: Read only %d (of %d) bytes.", len, nbl*4*128);
	  send_line(buf);
	  free(entry);
	  fclose(fp);
	  return 1;
	}
	fill_class_obs(&cobs, entry, nbl * 4 * 128);
#ifdef DEBUG
 	printf("%7d %12s %12s %d %d %d %d %d %d %d %d %d (%d %d %d %d | %d %d %d %d | %d %d %d %d)\n",
	 	m+1,
	 	c->xsourc, c->xline, c->xkind, 
	 	cobs.nbl, cobs.bytes, cobs.adr, cobs.nhead, cobs.len, cobs.ientry, cobs.nsec, cobs.obsnum,
	   	cobs.sec_cod[0], cobs.sec_cod[1], cobs.sec_cod[2], cobs.sec_cod[3],
	   	cobs.sec_adr[0], cobs.sec_adr[1], cobs.sec_adr[2], cobs.sec_adr[3],
	   	cobs.sec_len[0], cobs.sec_len[1], cobs.sec_len[2], cobs.sec_len[3]);
#endif
        
	kmax = cobs.nsec;
        if (kmax > 4) kmax = 4;
	
	for (k=0; k<kmax; k++) {
	  fill_class_header(cobs.sec_cod[k], cobs.sec_adr[k], cobs.sec_len[k],
			    entry, nbl*4*128, c);
	}
	
	/* Scan data for this entry*/
	fill_class_data(&cobs, entry, nbl*4*128, c);
	
	free(entry);
    }
    
#ifdef DEBUG
    printf("Return get_classfile_data() at nst=%d  nbl=%d  obsnum=%d\n", nst, cobs.nbl, cobs.obsnum);
#endif
    
    return 0;
}

int get_class(char *file, FDATA *fd)
{
    int i, j, nchan=0;
    CLASS *c;
    double sp, cp, sqrBt, tmp;
    float dat;
    
    void strip_trailing_spaces(char *);
    double rta(float);
    void MJDNtoDate(DATE *, int);
    int CheckDataSize(int);

    if (current < 0 || current >= nst) return -1;
    
    if (!st) return -2;
    
    c = &st[current];
    
    if (!c) return -3;
    
#ifdef DEBUG    
    printf("scan=%d (%d) [%s,%s]\n", current, nst, c->xsourc, c->xtel);
#endif

    strncpy(fd->sname, c->xsourc, 12);
    fd->sname[12] = '\0';
    strip_trailing_spaces(fd->sname);
    strncpy(fd->molecule, c->xline, 12);
    fd->molecule[12] = '\0';
    strip_trailing_spaces(fd->molecule);
    
#ifdef DEBUG
    printf("Step 1: [%s,%s]\n", fd->sname, fd->molecule);
#endif

    fd->n = (int)c->ndata;
    fd->sno = (int)c->xscan;
    fd->subno = 0;
    fd->coordType = 0;
    /* SetCoordType(s->CSystem); */
    /*
    if ((pa = CheckPosAngle()))
        fd->posang = (*pa)*PI/180.0;
    else
        fd->posang = (double)c->xposa;
    */
    fd->posang = 0.0;
    cp = cos(fd->posang);
    sp = sin(fd->posang);
    fd->xoff = rta(c->xoff1)*cp - rta(c->xoff2)*sp;
    fd->yoff = rta(c->xoff2)*cp + rta(c->xoff1)*sp;
    fd->equinox = c->p.epoch;
    if (fd->equinox <= 1950.1) {
        fd->epoch = 'B';
    } else {
        fd->epoch = 'J';
    }
    fd->x0 = c->p.lam;
    fd->y0 = c->p.bet;
/*    if (CheckCRVALType()) {
        fd->y0 -= s->BMapOff;
        fd->x0 -= s->LMapOff/cos(fd->y0);
    }
    fd->scanType = s->ObsMode;
 */
    fd->tsys = (double)c->g.tsys;
    fd->tau  = (double)c->g.tau;
    fd->taus = (double)c->c.taus;
    fd->taui = (double)c->c.taui;
    fd->int_time = (double)c->g.time;
    fd->vlsr = c->s.voff;
    MJDNtoDate(&(fd->date), c->xdobs + 60549);
    fd->date.Hour  = (int)(c->g.ut * RADTOHR);
    fd->date.Min   = (int)(60.0*(c->g.ut * RADTOHR - fd->date.Hour));
    fd->date.Sec   = (int)(60.0*(60.0*(c->g.ut * RADTOHR- fd->date.Hour) -
                           fd->date.Min));
    fd->LST = fd->date;
    fd->LST.Hour   = (int)(c->g.st * RADTOHR);
    fd->LST.Min    = (int)(60.0*(c->g.st * RADTOHR - fd->LST.Hour));
    fd->LST.Sec    = (int)(60.0*(60.0*(c->g.st * RADTOHR- fd->LST.Hour) -
                           fd->LST.Min));
    fd->az = c->g.az * RADTODEG;
    fd->el = c->g.el * RADTODEG;
    fd->aoff = fd->eoff = 0.0;
/*
    fd->aoff = rta(s->AzMapOff)*cp - rta(s->ElMapOff)*sp;
    fd->eoff = rta(s->ElMapOff)*cp + rta(s->AzMapOff)*sp;
  */
/*
#ifdef SISYFOS
    if (sscanf(s->Program, "COR%d", &arr_no) == 1) {
        if (arr_no >=1 && arr_no <= 4) {
            fd->aoff += (SisyfosAz[arr_no] - SisyfosAz[0]);
            fd->eoff += (SisyfosEl[arr_no] - SisyfosEl[0]);
            NorthAz = s->Azimuth + PI;
            GetEquOffsets(&(fd->LST), NorthAz, s->Elevation,
                          fd->aoff, fd->eoff,
                          s->Longitude, s->Latitude,
                          &RAOffset, &DecOffset);
            fd->xoff += RAOffset;
            fd->yoff += DecOffset;
        }
    }
#endif
#ifdef ONTHEFLY
    NorthAz = s->Azimuth + PI;
    GetEquOffsets(&(fd->LST), NorthAz, s->Elevation,
                  fd->aoff, fd->eoff,
                  s->Longitude, s->Latitude,
                  &RAOffset, &DecOffset);
    fd->xoff += RAOffset;
    fd->yoff += DecOffset;
#endif
*/
    fd->b.maj = 0;
    fd->b.min = 0;
    fd->b.PA  = 0;
    fd->beameff = c->c.beeff;
    fd->pol     = 0;
    fd->TAir = c->c.tamb; 
    fd->PAir = c->c.pamb; 
    fd->RAir = c->c.h2omm; 
    
    fd->firstIF = 6000.0/1000.0;
    fd->skyfreq = 0.0/1000.0;
    
    if (c->xkind == 0)
       nchan = c->s.nchan;
    else
       nchan = c->u.npoin;
       
    if (CheckDataSize(nchan) < nchan) {
        sprintf(buf, "Out of memory: Can't allocate enough memory for NChan=%d.", nchan);
        send_line(buf);
	return -4;
    }
    
/* APEX SEPIA stuff */
/*     GARD 180, Band 5 */
    if (strcmp("AP-G101-XF01", c->xtel) == 0) {
        fd->pol = 11;
    } else if (strcmp("AP-G101-XF02", c->xtel) == 0) {
        fd->pol = 12;
    } else if (strcmp("AP-G102-XF03", c->xtel) == 0) {
        fd->pol = 23;
    } else if (strcmp("AP-G102-XF04", c->xtel) == 0) {
        fd->pol = 24;
    } else if (strcmp("AP-G103-XF05", c->xtel) == 0) {
        fd->pol = 15;
    } else if (strcmp("AP-G103-XF06", c->xtel) == 0) {
        fd->pol = 16;
    } else if (strcmp("AP-G104-XF07", c->xtel) == 0) {
        fd->pol = 27;
    } else if (strcmp("AP-G104-XF08", c->xtel) == 0) {
        fd->pol = 28;
    }
    if (strcmp("AP-S101-XF01", c->xtel) == 0 || strcmp("AP-S101-PF01", c->xtel) == 0) {
        fd->pol = 11;
    } else if (strcmp("AP-S101-XF02", c->xtel) == 0 || strcmp("AP-S101-PF02", c->xtel) == 0) {
        fd->pol = 12;
    } else if (strcmp("AP-S102-XF03", c->xtel) == 0 || strcmp("AP-S102-PF03", c->xtel) == 0) {
        fd->pol = 23;
    } else if (strcmp("AP-S102-XF04", c->xtel) == 0 || strcmp("AP-S102-PF04", c->xtel) == 0) {
        fd->pol = 24;
    } else if (strcmp("AP-S103-XF05", c->xtel) == 0 || strcmp("AP-S103-PF05", c->xtel) == 0) {
        fd->pol = 15;
    } else if (strcmp("AP-S103-XF06", c->xtel) == 0 || strcmp("AP-S103-PF06", c->xtel) == 0) {
        fd->pol = 16;
    } else if (strcmp("AP-S104-XF07", c->xtel) == 0 || strcmp("AP-S104-PF07", c->xtel) == 0) {
        fd->pol = 27;
    } else if (strcmp("AP-S104-XF08", c->xtel) == 0 || strcmp("AP-S104-PF08", c->xtel) == 0) {
        fd->pol = 28;
    }
/*     NOVA 690, Band 9 */
    if (strcmp("AP-N601-XF01", c->xtel) == 0 || strcmp("AP-N601-PF01", c->xtel) == 0) {
        fd->pol = 61;
    } else if (strcmp("AP-N601-XF02", c->xtel) == 0 || strcmp("AP-N601-PF02", c->xtel) == 0) {
        fd->pol = 62;
    } else if (strcmp("AP-N602-XF03", c->xtel) == 0 || strcmp("AP-N602-PF03", c->xtel) == 0) {
        fd->pol = 63;
    } else if (strcmp("AP-N602-XF04", c->xtel) == 0 || strcmp("AP-N602-PF04", c->xtel) == 0) {
        fd->pol = 64;
    }
/* NOVA 2SB, Band 9 */
    if (strcmp("AP-S601-XF01", c->xtel) == 0 || strcmp("AP-S601-PF01", c->xtel) == 0) {
        fd->pol = 61;
    } else if (strcmp("AP-S601-XF02", c->xtel) == 0 || strcmp("AP-S601-PF02", c->xtel) == 0) {
        fd->pol = 62;
    } else if (strcmp("AP-S602-XF03", c->xtel) == 0 || strcmp("AP-S602-PF03", c->xtel) == 0) {
        fd->pol = 73;
    } else if (strcmp("AP-S602-XF04", c->xtel) == 0 || strcmp("AP-S602-PF04", c->xtel) == 0) {
        fd->pol = 74;
    } else if (strcmp("AP-S603-XF05", c->xtel) == 0 || strcmp("AP-S603-PF05", c->xtel) == 0) {
        fd->pol = 65;
    } else if (strcmp("AP-S603-XF06", c->xtel) == 0 || strcmp("AP-S603-PF06", c->xtel) == 0) {
        fd->pol = 66;
    } else if (strcmp("AP-S604-XF07", c->xtel) == 0 || strcmp("AP-S604-PF07", c->xtel) == 0) {
        fd->pol = 77;
    } else if (strcmp("AP-S604-XF08", c->xtel) == 0 || strcmp("AP-S604-PF08", c->xtel) == 0) {
        fd->pol = 78;
    }
/* NOVA 2SB, Band 9, new 2019 IF, new 2020  */
    if (strcmp("AP-S601-F101", c->xtel) == 0 || strcmp("AP-S601-PF01", c->xtel) == 0 || strcmp("AP-S301-F103", c->xtel) == 0) {
        fd->pol = 81;
    } else if (strcmp("AP-S601-F105", c->xtel) == 0 || strcmp("AP-S601-PF02", c->xtel) == 0 || strcmp("AP-S301-F101", c->xtel) == 0) {
        fd->pol = 82;
    } else if (strcmp("AP-S602-F106", c->xtel) == 0 || strcmp("AP-S602-PF03", c->xtel) == 0 || strcmp("AP-S302-F103", c->xtel) == 0) {
        fd->pol = 93;
    } else if (strcmp("AP-S602-F102", c->xtel) == 0 || strcmp("AP-S602-PF04", c->xtel) == 0 || strcmp("AP-S302-F101", c->xtel) == 0) {
        fd->pol = 94;
    } else if (strcmp("AP-S603-F103", c->xtel) == 0 || strcmp("AP-S603-PF05", c->xtel) == 0 || strcmp("AP-S303-F104", c->xtel) == 0) {
        fd->pol = 85;
    } else if (strcmp("AP-S603-F107", c->xtel) == 0 || strcmp("AP-S603-PF06", c->xtel) == 0 || strcmp("AP-S303-F102", c->xtel) == 0) {
        fd->pol = 86;
    } else if (strcmp("AP-S604-F104", c->xtel) == 0 || strcmp("AP-S604-PF07", c->xtel) == 0 || strcmp("AP-S304-F104", c->xtel) == 0) {
        fd->pol = 97;
    } else if (strcmp("AP-S604-F108", c->xtel) == 0 || strcmp("AP-S604-PF08", c->xtel) == 0 || strcmp("AP-S304-F102", c->xtel) == 0) {
        fd->pol = 98;
    }
    
#ifdef DEBUG
    printf("pol,tel=%d, %s\n", fd->pol, c->xtel);
#endif
    
    if (c->xkind == 0) { /* Spectral line scan */
      fd->lofreq = (c->s.restf + c->s.image)/2/1000.0;
      fd->f0 = (c->s.restf + (0.0 - c->s.rchan)*c->s.fres)/1000.0;
      fd->fn = (c->s.restf + (c->s.nchan - 1.0 - c->s.rchan)*c->s.fres)/1000.0;
      fd->fres = c->s.fres;
      fd->v0 = c->s.voff + (0.0 - c->s.rchan)*c->s.vres;
      fd->vres = c->s.vres;
      if (c->s.fres < 0.0) { /* always store in fd with positive df */
        fd->fres = -fd->fres;
        tmp = fd->f0; fd->f0 = fd->fn; fd->fn = tmp;
	fd->vres = -fd->vres;
	fd->v0 = c->s.voff + (c->s.nchan - 1.0 - c->s.rchan)*c->s.vres;
      }
      sqrBt = sqrt(fabs(c->s.fres) * 1.0e6 * fd->int_time);
      for (i=0; i<c->s.nchan; i++) {
          j = i;
	  if (c->s.fres < 0.0) j = c->s.nchan - 1 - i;
	  dat = c->data[j];
	  /* if (!isfinite(dat)) {
	    sprintf(buf,
	      "Warning. %d %s %s contains infinite floating point data point at channel %d. Set to zero.",
	      c->xscan, c->xline, c->xsourc, j);
	    send_line(buf);
	    dat = 0.0;
	  } */
          fd->d[i] = dat;
	  if (sqrBt > 0.0)
	    fd->e[i] = 2.0*fd->tsys/sqrBt;
	  else
	    fd->e[i] = 0.0;
      }
      /* Check this here: */
      fd->firstIF = (c->s.restf - c->s.image)/2./1000.0;
      fd->skyfreq = c->s.restf*(1.0 - c->s.doppler/SPEEDOFLIGHT)/1000.0;
      /* sprintf(buf, "Source: '%s'  Vdop=%f km/s", c->xsourc, c->s.doppler);
      send_line(buf); */
    } else { /* continuum scan */
      fd->lofreq = (c->u.freq + c->u.cimag)/2/1000.0;
      fd->f0 = (c->u.tref + (0.0 - c->u.rpoin)*c->u.tres)/1000.0;
      fd->fn = (c->u.tref + (c->u.npoin - 1 - c->u.rpoin)*c->u.tres)/1000.0;
      fd->fres = c->u.tres;
      fd->v0 = rta(c->u.aref + (0.0 - c->u.rpoin)*c->u.ares);
      fd->vres = rta(c->u.ares);
      sqrBt = sqrt(fabs(c->u.width) * 1.0e6 * fd->int_time);
      for (i=0; i<c->u.npoin; i++) {
          fd->d[i] = (double)c->data[i];
	  if (sqrBt > 0.0)
	    fd->e[i] = 2.0*fd->tsys/sqrBt;
	  else
	    fd->e[i] = 0.0;
      }
    }
/*    if (s->FreqRes < 0.0) {
        fd->f0 = Frequency(x->NChannel-1, s, x)/1000.;
        fd->fn = Frequency(0, s, x)/1000.;
        if (odin) {
            fd->v0 = VelOdin(x->NChannel-1, s, x);
        } else {
            fd->v0 = Velocity(x->NChannel-1, s, x);
        }
        fd->fres = -s->FreqRes;
        fd->vres = -s->VelRes;
        sqrBt = sqrt(fabs(fd->fres) * fd->int_time)*1000.0;
        for (i=x->NChannel-1; i>=0; i--) {
            fd->d[x->NChannel-1-i] = (double)(s->c[i]);
            if (sqrBt > 0.0 && fd->tsys > 0.0)
                fd->e[x->NChannel-1-i] = 2.0*fd->tsys/sqrBt;
            else
                fd->e[x->NChannel-1-i] = 1.0;
        }
    } else {
        fd->fn = Frequency(x->NChannel-1, s, x)/1000.;
        fd->f0 = Frequency(0, s, x)/1000.;
        if (odin) {
            fd->v0 = VelOdin(0, s, x);
        } else {
            fd->v0 = Velocity(0, s, x);
        }
        fd->fres = s->FreqRes;
        fd->vres = s->VelRes;
        sqrBt = sqrt(fabs(fd->fres) * fd->int_time)*1000.0;
        for (i=0; i<x->NChannel; i++) {
          fd->d[i] = (double)(s->c[i]);
          if (sqrBt > 0.0 && fd->tsys > 0.0)
              fd->e[i] = 2.0*fd->tsys/sqrBt;
          else
              fd->e[i] = 1.0;
        }
    }
    */
#ifdef DEBUG
    printf("End. current, nst: %d %d.\n", current, nst);
#endif
    return 0;
}

static void LoadClassScans(char *file, int nfound, CLASS_INFO *i, int extno)
{
    int ierr, n, nread=0, nfirst=1, nend=nfound;
    DataSetPtr d;
    
    list   *get_listlist();
    DataSetPtr new_dataset(list *, char *, DataSetPtr);
    void DeleteLastDataSet();
    int count_scans(DataSetPtr);
    void obtain_map_info(Widget, char *, XtPointer);
    void UpdateData(int, int);
    int read_file(char *, char *, DataSetPtr);
    char *StripSuffix(const char *);
    char *StripPath(const char *);
    
    
    if (i->type == 2) nfirst=0;
    if (i->type == 1) nend = nfound+1;
    
    if (file)
       sprintf(buf, "CLASSv%d %s ext %d", i->type, StripSuffix(StripPath(file)), extno);
    else
       strcpy(buf, "CLASS file");
       
    d = new_dataset(get_listlist(), buf, NULL);
    
    if (!d) {
        sprintf(buf, "Couldn't allocate dataset for %s.", file);
	send_line(buf);
        return;
    }
	
#ifdef DEBUG    
    printf("Trying to load %d CLASS scans from %s.\n", nfound, file);
#endif
    for (n=nfirst; n<nend; n++) {
        current = n;
	if (n==nfirst)
	  ierr = read_file("seqclass", file, d);
	else
	  ierr = read_file("seqclass", file, NULL);
	  
	if (ierr == 0) nread++;
    }
    sprintf(buf, "Read %d scans (of %d) from CLASS file %s.",
            nread, nfound, StripPath(file));
    send_line(buf);
    if (count_scans(d) > 0) {
        vP->from = d;
        if (count_scans(d) > 1) {
            obtain_map_info(NULL, "map", NULL);
        }
	UpdateData(SCALE_BOTH, REDRAW);
    } else {
        vP->to = vP->from;
	DeleteLastDataSet();
    }
#ifdef DEBUG
    printf("Loading done.\n");
#endif

}

int PostClassScanListDialog(char *file, Widget w, fsel_struct *fs)
{
    int ierr=1, n=0, nex, m;
    CLASS_INFO info;
    FILE *fp = NULL;

    info.type = 0;
    
    set_classfile_type(file, &info);
    
    if (info.type == 0) {
        sprintf(buf, "Unknown CLASS file type in %s.", file);
	send_line(buf);
        return 0;
    }
        
#ifdef DEBUG    
    printf("File: '%s'  type=%d\n", file, info.type);
#endif
    
    fp = get_classfile_descriptor(file, &info);
    
    if (!fp) {
        return 0;
    }
    
    if (info.type == 1) {
        n = get_class_v1_file_listing(fp, &info);
        if (n <= 0) {
            if (st) free(st);
	    st = NULL;
	    nst = 0;
            return 0;
        }
        ierr = get_classfile_data(fp, n, &info);
        if (!ierr) {
            LoadClassScans(file, n, &info, 0);
        }
#ifdef DEBUG
        printf("1 Freed st[] array for %d scans.\n", nst);
#endif    
        if (st) {
            for (n=1; n<nst; n++)
                if (st[n].ndata > 0 && st[n].data) free(st[n].data);
            free(st);
	    st = NULL;
	    nst = 0;
        }
    } else if (info.type == 2) {
        nex = info.nex;
        for (m=0; m<nex; m++) {
            n = get_class_v2_file_listing(fp, &info, m);
            if (n <= 0) {
                if (st) free(st);
	        st = NULL;
	        nst = 0;
                return 0;
            }
            ierr = get_classfile_v2_data(fp, n, &info, m);
            if (!ierr) {
                LoadClassScans(file, n, &info, m);
            }
            if (st) {
#ifdef DEBUG
    printf("2 Freed st[] array for %d scans.\n", nst);
#endif    
               for (n=0; n<nst; n++)
                   if (st[n].ndata > 0 && st[n].data) free(st[n].data);
               free(st);
	       st = NULL;
	       nst = 0;
            }
	}
    }
    
    return 0;
}
