/**************************** XS ********************************************
Copyright (C) 2000-2023  P. Bergman

This program is free software; you can redistribute it and/or modify it under 
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/
#include <stdio.h>
#include <math.h>

#include <Xm/Text.h>
#include <Xm/Label.h>
#include <Xm/RowColumn.h>
#include <Xm/Frame.h>
#include <Xm/ToggleB.h>

#include "defines.h"
#include "global_structs.h"
#include "dialogs.h"

/*** External variables ***/
extern VIEW  *vP;

void ManageDialogCenteredOnPointer(Widget);
void PostErrorDialog(Widget, char *);

list       scan_iterator(list, DataSetPtr);
scanPtr    copy_scanheader(DataSetPtr, int, scanPtr);
list      *get_listlist();
DataSetPtr new_dataset(list *, char *, DataSetPtr);
void       DeleteLastDataSet();

/*** Local variables ***/
typedef struct {
    double k, m, JyK, el;
    int km;
    int beam_eff;
    int jyk_eff;
} LINE;

static LINE scale;

void init_scale_parameters()
{
    scale.k = 1.0;
    scale.m = 0.0;
    scale.km = 1;
    scale.beam_eff = 0;
    scale.jyk_eff = 0;
/* OSO 20m values Feb 2019 from Uranus scans */
    scale.JyK = 23.0;
    scale.el = 38.0;
}

/* From OSO 20m R Cas measurements 2019-02-20 at 86 GHz */
static double jyk_function(double e)
{
    double arg, f;
    
    arg = (e - 64.488)/51.118;
    f = 16.724 + 9.425 * exp(-ALPHA*arg*arg);
    
    return f;
}

static scanPtr scale_spectrum(DataSetPtr d, scanPtr s, LINE *L)
{
    int n;
    double corr, fel, fnorm;
    scanPtr new = NULL;
    
    new = copy_scanheader(d, s->nChan, s);
    if (!new) return new;
    
    for (n=0; n<new->nChan; n++) {
        if (L->beam_eff) {
            if (s->beameff > 0.0 && s->beameff <= 1.0) {
                new->d[n] = s->d[n]/s->beameff;
                new->e[n] = s->e[n]/s->beameff;
            }
        } else if (L->jyk_eff) {
	    fel = jyk_function(s->el);
	    fnorm = jyk_function(L->el);
	    corr = L->JyK * fnorm/fel;
            new->d[n] = s->d[n]*corr;
            new->e[n] = s->e[n]*corr;
	} else {
            new->d[n] = L->k * s->d[n] + L->m;
            if (new->fft)
                new->e[n] = L->k * s->e[n] + L->m;
            else
                new->e[n] = L->k * s->e[n];
        }
    }
    
    return new;
}

static void get_scale_strs(Widget w, StdForm *sf, XmAnyCallbackStruct *cb)
{
    int n = 0;
    string buf, name;
    list curr = NULL;
    scanPtr s, new;
    DataSetPtr dsp;

    void UpdateData(int, int), send_line(char *), wdscanf(Widget, double *);

    wdscanf(sf->edit[0], &scale.k);
    wdscanf(sf->edit[1], &scale.m);

    if (scale.beam_eff) {
        sprintf(buf, "Dividing all spectra with the header main beam efficiency.");
        sprintf(name, "Scaled (1/eta_mb)");
    } else if (scale.jyk_eff) {
        sprintf(buf, "Scale all spectra with the OSO K to Jy elevation function.");
        sprintf(name, "Scaled (Jy/K)");
    } else {
        if (scale.m >= 0.0) {
            if (scale.m == 0.0) {
                sprintf(buf, "Scaling all spectra with y' = %g*y",
                        scale.k);
                sprintf(name, "Scaled (%g*y)", scale.k);
            } else {
                sprintf(buf, "Scaling all spectra with y' = %g*y + %g",
                        scale.k, scale.m);
                sprintf(name, "Scaled (%g*y+%g)", scale.k, scale.m);
            }
        } else {
            sprintf(buf, "Scaling all spectra with y' = %g*y - %g",
                    scale.k, -scale.m);
            sprintf(name, "Scaled (%g*y-%g)", scale.k, -scale.m);
        }
    }
    send_line(buf);
    
    dsp = new_dataset(get_listlist(), name, vP->from);
    if (!dsp) return;

    if (vP->mode == SHOW_SPE) {
        new = scale_spectrum(dsp, vP->s, &scale);
        if (!new) {
            PostErrorDialog(w, "Out of memory when allocating new scan.");
            DeleteLastDataSet();
        } else {
            vP->to = vP->from = dsp;
            vP->s = new;
            UpdateData(SCALE_ONLY_Y, REDRAW);
        }
        return;
    }
    
    n = 0;
    while ( (curr = scan_iterator(curr, vP->from)) != NULL) {
        s = (scanPtr)DATA(curr);
        new = scale_spectrum(dsp, s, &scale);
        if (!new) break;
        n++;
    }
    
    if (n == 0) {
        PostErrorDialog(w, "No new scans created.");
        DeleteLastDataSet();
        return;
    } else {
        vP->to = vP->from = dsp;
        vP->s = (scanPtr)DATA(dsp->scanlist);
    }

    UpdateData(SCALE_ONLY_Y, REDRAW);
}

static void set_km_active(Widget w, char *str,
                            XmToggleButtonCallbackStruct *cb)
{
    /* if (cb->set) { */
        scale.km = 1;
	scale.jyk_eff = 0;
        scale.beam_eff = 0;
    /* } else {
        scale.beam_eff = 0;
    } */
}

static void set_beff_active(Widget w, char *str,
                            XmToggleButtonCallbackStruct *cb)
{
    /* if (cb->set) { */
        scale.beam_eff = 1;
	scale.jyk_eff = 0;
	scale.km = 0;
    /* } else {
        scale.beam_eff = 0;
    } */
}

static void set_jyk_active(Widget w, char *str,
                            XmToggleButtonCallbackStruct *cb)
{
    /* if (cb->set) { */
	scale.jyk_eff = 1;
	scale.beam_eff = 0;
	scale.km = 0;
    /* } else {
	scale.jyk_eff = 0;
    } */
}

void scale_scans(Widget wid, char *cmd, XtPointer call_data)
{
    int n;
    Widget rc, rcV, fr, rb, kmButt, beffButt, jykButt;
    Widget w = wid;
    Arg wargs[10];
    StdForm *sf;
    string s0, s1;

    void w_printf(Widget, char *, ...);

    while (!XtIsWMShell(w))
        w = XtParent(w);

    sf = PostStdFormDialog(w, "Scale spectra",
             BUTT_APPLY, (XtCallbackProc)get_scale_strs, NULL,
             BUTT_CANCEL, NULL, NULL,
             NULL, NULL, NULL,
             2, NULL);

    rcV = XtVaCreateManagedWidget("rowcol", xmRowColumnWidgetClass, sf->form,
                                  XmNorientation,       XmVERTICAL,
                                  NULL);

    fr = XtVaCreateWidget("frame", xmFrameWidgetClass, rcV,
				          XmNshadowType, XmSHADOW_OUT, NULL);
    rc = XtVaCreateManagedWidget("rowcol", xmRowColumnWidgetClass, fr,
                                 XmNorientation,       XmHORIZONTAL,
                                 NULL);
    XtCreateManagedWidget("y' = ", xmLabelWidgetClass, rc, NULL, 0);
    sf->edit[0] = XtCreateManagedWidget("edit", xmTextWidgetClass,
                                        rc, NULL, 0);
    XtCreateManagedWidget("y + ", xmLabelWidgetClass, rc, NULL, 0);
    sf->edit[1] = XtCreateManagedWidget("edit", xmTextWidgetClass,
                                        rc, NULL, 0);
    n = 0;
    XtSetArg(wargs[n], XmNentryClass, xmToggleButtonWidgetClass); n++;
    rb = XmCreateRadioBox(rcV, "radiobox", wargs, n);
    
    n = 0;
    if (scale.km == 1) {
        XtSetArg(wargs[n], XmNset, True); n++;
    }
    kmButt = XtCreateWidget("Scale as above",
                              xmToggleButtonWidgetClass,
                              rb, wargs, n);
    /* XtAddCallback(beffButt, XmNvalueChangedCallback, */
    XtAddCallback(kmButt, XmNarmCallback,
                  (XtCallbackProc)set_km_active, NULL);
    n = 0;
    if (scale.beam_eff == 1) {
        XtSetArg(wargs[n], XmNset, True); n++;
    }
    beffButt = XtCreateWidget("Divide by header main beam efficiency",
                              xmToggleButtonWidgetClass,
                              rb, wargs, n);
    /* XtAddCallback(beffButt, XmNvalueChangedCallback, */
    XtAddCallback(beffButt, XmNarmCallback,
                  (XtCallbackProc)set_beff_active, NULL);

    n = 0;
    if (scale.jyk_eff == 1) {
        XtSetArg(wargs[n], XmNset, True); n++;
    }
    jykButt = XtCreateWidget("Divide by OSO20m JyK function",
                              xmToggleButtonWidgetClass,
                              rb, wargs, n);
    /* XtAddCallback(jykButt, XmNvalueChangedCallback, */
    XtAddCallback(jykButt, XmNarmCallback,
                  (XtCallbackProc)set_jyk_active, NULL);

    ArrangeStdFormDialog(sf, fr);

    sprintf(s0, "%f", scale.k);
    sprintf(s1, "%f", scale.m);

//    w_printf(sf->edit[0], "%f", scale.k);
//    w_printf(sf->edit[1], "%f", scale.m);

    w_printf(sf->edit[0], s0);
    w_printf(sf->edit[1], s1);

    XtManageChild(kmButt);
    XtManageChild(beffButt);
    XtManageChild(jykButt);
    XtManageChild(rb);
    XtManageChild(fr);
    
    ManageDialogCenteredOnPointer(sf->form);
}
